//	TorusGamesGraphicsViewiOS.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesGraphicsViewiOS.h"
#import "TorusGamesRenderer.h"
#import "TorusGames-Common.h"
#import "TorusGamesFlickGestureRecognizer.h"
#import "TorusGamesTwoFingerGestureRecognizer.h"
#import "GeometryGamesModel.h"
#import "GeometryGamesUtilities-Common.h"
#import "GeometryGamesUtilities-Mac-iOS.h"


//	Privately-declared methods
@interface TorusGamesGraphicsViewiOS()
- (void)handleTwoFingerDrag:(TorusGamesTwoFingerGestureRecognizer *)aTwoFingerGestureRecognizer;
- (void)handleFlickGesture:(TorusGamesFlickGestureRecognizer *)aFlickGesture;
- (CGPoint)touchLocation:(UITouch *)aTouch;
- (CGPoint)gestureLocation:(UIGestureRecognizer *)aGestureRecognizer;
- (CGPoint)pointLocation:(CGPoint)aLocationInView;
- (CGPoint)touchMotion:(UITouch *)aTouch;
@end


@implementation TorusGamesGraphicsViewiOS
{
	//	Store the time of the previous touch motion,
	//	so we can later compute a velocity.
	double	itsLastTouchTime;
	
	//	-handleTwoFingerDrag will need to compute (Δh, Δv).
	CGPoint	itsPreviousTwoFingerGestureLocation;
}


- (id)initWithModel:(GeometryGamesModel *)aModel frame:(CGRect)aFrame
{
	self = [super initWithModel:aModel frame:aFrame];
	if (self != nil)
	{
		CALayer									*theLayer;
		TorusGamesTwoFingerGestureRecognizer	*theTwoFingerRecognizer;
		TorusGamesFlickGestureRecognizer		*theFlickRecognizer;


		theLayer = [self layer];

		GEOMETRY_GAMES_ASSERT(
			[theLayer isKindOfClass:[CAMetalLayer class]],
			"Internal error:  layer is not CAMetalLayer");
		
		itsRenderer = [[TorusGamesRenderer alloc]
			initWithLayer:	(CAMetalLayer *)theLayer
			device:			MTLCreateSystemDefaultDevice()	//	on all iOS devices, this is the integrated GPU
			multisampling:	true	//	On the one hand, in the 2D games
									//		all the antialiasing takes place
									//		in the texture filtering in the sprites' interiors
									//		(no polygon edges are visible),
									//		so there's no reason to use multisampling there.
									//	On the other hand, the 3D games will
									//		need multisampling to look their best.
			depthBuffer:	true	//	Depth buffer needed for 3D games.
			stencilBuffer:	false];

		theTwoFingerRecognizer = [[TorusGamesTwoFingerGestureRecognizer alloc]
								initWithTarget:	self
								action:			@selector(handleTwoFingerDrag:)];
		[self addGestureRecognizer:theTwoFingerRecognizer];

		theFlickRecognizer = [[TorusGamesFlickGestureRecognizer alloc]
								initWithTarget:	self
								action:			@selector(handleFlickGesture:)];
		[self addGestureRecognizer:theFlickRecognizer];
	}
	return self;
}

- (void)layoutSubviews
{
	ModelData	*md	= NULL;

	//	In the case of a GeometryGamesGraphicsViewiOS, a call to -layoutSubviews
	//	has absolutely nothing to do with subviews, because the view has no subviews.
	//	Rather the call is telling us that the view's dimensions may have changed.

	//	Let the GeometryGamesGraphicsViewiOS implementation of this method resize the framebuffer.
	[super layoutSubviews];

	//	Request a redraw.
	[itsModel lockModelData:&md];
	md->itsChangeCount++;
	[itsModel unlockModelData:&md];
}


#pragma mark -
#pragma mark renderer refresh

- (void)refreshRendererTexturesWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesWithModelData:md];
}

- (void)refreshRendererTexturesForGameResetWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesForGameResetWithModelData:md];
}

- (void)refreshRendererTexturesForCharacterInputWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesForCharacterInputWithModelData:md];
}


#pragma mark -
#pragma mark touch handling

- (void)touchesBegan:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	UITouch		*theTouch;
	CGPoint		theTouchPoint;
	ModelData	*md	= NULL;

	UNUSED_PARAMETER(event);
	
	//	By default UIView processes only single touches,
	//	so we're guaranteed to get information about
	//	the first finger to touch the display.
	//	UIView suppresses information about any other fingers
	//	that touch the display as part of the same gesture.
	theTouch = [touches anyObject];
	
	//	Where did theTouch begin?
	theTouchPoint = [self touchLocation:theTouch];

	//	Pass the touch-began to the platform-independent code.
	[itsModel lockModelData:&md];
	MouseDown(	md,
				theTouchPoint.x,
				theTouchPoint.y,
				[theTouch timestamp],
				false,	//	aScrollFlag (used in desktop version only)
				false,	//	aMarkFlag
				false,	//	aTemporalDoubleClick (used in desktop version only)
				false,	//	aTwoFingerGestureFlag
				false);	//	aFlickGestureFlag
	[itsModel unlockModelData:&md];

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
	[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
}

- (void)touchesMoved:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	UITouch		*theTouch;
	CGPoint		theTouchMotion;
	ModelData	*md	= NULL;

	UNUSED_PARAMETER(event);

	//	By default, a UIView gets events for only the first touch,
	//	so [touches anyObject] always returns the same touch.
	theTouch = [touches anyObject];

	//	How far has the user's finger moved since the previous
	//	call to touchesBegan or touchesMoved?
	theTouchMotion = [self touchMotion:theTouch];

	//	Pass the touch-motion to the platform-independent code.
	[itsModel lockModelData:&md];
	MouseMove(	md,
				theTouchMotion.x,
				theTouchMotion.y,
				[theTouch timestamp],
				false,	//	aTwoFingerGestureFlag
				false);	//	aFlickGestureFlag
	[itsModel unlockModelData:&md];

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
	[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
}

- (void)touchesEnded:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	UITouch				*theTouch;
	ModelData			*md				= NULL;
	TitledErrorMessage	theDragError;
	NSString			*theTitle,
						*theMessage;
	UIAlertController	*theAlertController;
	UIViewController	*thePresentingController;

	UNUSED_PARAMETER(event);

	//	By default, a UIView gets events for only the first touch,
	//	so [touches anyObject] always returns the same touch.
	theTouch = [touches anyObject];

	[itsModel lockModelData:&md];

	MouseUp(md,
			[theTouch timestamp],
			false,	//	aTwoFingerGestureFlag
			false,	//	aFlickGestureFlag
			false);	//	aTouchSequenceWasCancelled

	theDragError		= md->itsDragMessage;
	md->itsDragMessage	= (TitledErrorMessage) {NULL, NULL};

	[itsModel unlockModelData:&md];

#ifdef SCIENCE_CENTER_VERSION
#error Before releasing another Science Center version,			\
	test whether the "place Gomoko stones at intersections"		\
	works OK with it.  In particular, if that message appears	\
	and then the user walks away, what happens when				\
	the exhibits times out and resets itself?
#endif
	//	Did MouseDown() ask us to display a message?
	if (theDragError.itsMessage != NULL
	 && theDragError.itsTitle   != NULL)
	{
		theTitle	= GetNSStringFromZeroTerminatedString(theDragError.itsTitle);
		theMessage	= GetNSStringFromZeroTerminatedString(theDragError.itsMessage);

		theAlertController = [UIAlertController
			alertControllerWithTitle:	theTitle
			message:					theMessage
			preferredStyle:				UIAlertControllerStyleAlert];

		[theAlertController addAction:[UIAlertAction
			actionWithTitle:	LocalizationNotNeeded(@"OK")
			style:				UIAlertActionStyleDefault
			handler:			nil]];

		//	Kludge warning:  This seems like a dicey way to find thePresentingController.
		//
		//	The clean way to do it would be to give the TorusGamesGraphicsViewiOS
		//	a delegate (which in practice would be the view controller)
		//	and let the delegate display the message.  Seems like overkill,
		//	though, for a message that's needed only for Gomoku, and even there
		//	is needed only rarely.
		//
		thePresentingController = [[self window] rootViewController];
		while ([thePresentingController presentedViewController] != nil)
		{
			thePresentingController = [thePresentingController presentedViewController];
		}

		[thePresentingController
			presentViewController:	theAlertController
			animated:				YES
			completion:				nil];
	}

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
	[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
}

- (void)touchesCancelled:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	//	touchesCancelled may get called for various reasons:
	//
	//	 - When a UIGestureRecognizer recognizes a gesture,
	//		and touches have not been delayed, the UIKit
	//		calls touchesCancelled to give us an opportunity
	//		to undo whatever we might have done in response
	//		to those touch events that were subsequently found
	//		to comprise a gesture.
	//
	//	 - With split-screen multitasking, if the UIKit
	//		reinterprets an already-underway touch sequence
	//		as a motion of the screen's dividing bar,
	//		it calls touchesCancelled.
	//
	//	 - If an incoming phone call interrupts a touch sequence,
	//		the UIKit calls touchesCancelled.
	//

	UITouch		*theTouch;
	ModelData	*md	= NULL;

	UNUSED_PARAMETER(event);

	//	By default, a UIView gets events for only the first touch,
	//	so [touches anyObject] always returns the same touch.
	theTouch = [touches anyObject];

	//	Pass the touch-cancelled to the platform-independent code.
	[itsModel lockModelData:&md];
	MouseCancelled(md, [theTouch timestamp]);
	[itsModel unlockModelData:&md];

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
	[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
}


#pragma mark -
#pragma mark two-finger drag gesture

- (void)handleTwoFingerDrag:(TorusGamesTwoFingerGestureRecognizer *)aTwoFingerGestureRecognizer
{
	CGPoint		theGestureLocation,
				theGestureMotion;
	ModelData	*md	= NULL;

	switch ([aTwoFingerGestureRecognizer state])
	{
		case UIGestureRecognizerStatePossible:
			break;
		
		case UIGestureRecognizerStateBegan:
			theGestureLocation = [self gestureLocation:aTwoFingerGestureRecognizer];
			[itsModel lockModelData:&md];
			MouseDown(	md,
						theGestureLocation.x,
						theGestureLocation.y,
						CFAbsoluteTimeGetCurrent(),
						false,	//	aScrollFlag (used in desktop version only)
						false,	//	aMarkFlag
						false,	//	aTemporalDoubleClick (used in desktop version only)
						true,	//	aTwoFingerGestureFlag
						false);	//	aFlickGestureFlag
			[itsModel unlockModelData:&md];
			itsPreviousTwoFingerGestureLocation	= theGestureLocation;
			break;

		case UIGestureRecognizerStateChanged:
			theGestureLocation = [self gestureLocation:aTwoFingerGestureRecognizer];
			[itsModel lockModelData:&md];
			theGestureMotion.x = theGestureLocation.x - itsPreviousTwoFingerGestureLocation.x;
			theGestureMotion.y = theGestureLocation.y - itsPreviousTwoFingerGestureLocation.y;
			MouseMove(	md,
						theGestureMotion.x,
						theGestureMotion.y,
						CFAbsoluteTimeGetCurrent(),
						true,	//	aTwoFingerGestureFlag
						false);	//	aFlickGestureFlag
			[itsModel unlockModelData:&md];
			itsPreviousTwoFingerGestureLocation	= theGestureLocation;
			break;
		
		case UIGestureRecognizerStateCancelled:
			//	Unlike a raw touch sequence, which gets cancelled
			//	when a gesture begins, a gesture should rarely get cancelled
			//	(perhaps just for an incoming phone call, or maybe
			//	with split-screen multitasking on iPad).
			//	If a two-finger drag gets cancelled, handle it
			//	the same as we'd handle UIGestureRecognizerStateEnded.
			//
			//	Fall through...
		case UIGestureRecognizerStateEnded:
			[itsModel lockModelData:&md];
			MouseUp(md,
					CFAbsoluteTimeGetCurrent(),
					true,	//	aTwoFingerGestureFlag
					false,	//	aFlickGestureFlag
					false);	//	aTouchSequenceWasCancelled
			[itsModel unlockModelData:&md];
			itsPreviousTwoFingerGestureLocation	= CGPointZero;
			break;
			
		case UIGestureRecognizerStateFailed:
			break;
	}

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
	[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
}


#pragma mark -
#pragma mark flick gesture

- (void)handleFlickGesture:(TorusGamesFlickGestureRecognizer *)aFlickGesture
{
	CGPoint		theStartPoint,
				theDisplacement;
	double		theStartTime,
				theDuration;
	ModelData	*md	= NULL;

	if ([aFlickGesture state] == UIGestureRecognizerStateRecognized)	//	should never fail
	{
		theStartPoint	= [aFlickGesture startPoint		];
		theDisplacement	= [aFlickGesture displacement	];
		theStartTime	= [aFlickGesture startTime		];
		theDuration		= [aFlickGesture duration		];

		//	A flick gesture is discrete:  we get notified once
		//	as soon as it's recognized, then never again.

		[itsModel lockModelData:&md];

		MouseDown(	md,
					theStartPoint.x,
					theStartPoint.y,
					theStartTime,
					false,	//	aScrollFlag (used in desktop version only)
					false,	//	aMarkFlag
					false,	//	aTemporalDoubleClick (used in desktop version only)
					false,	//	aTwoFingerGestureFlag
					true);	//	aFlickGestureFlag

		MouseMove(	md,
					theDisplacement.x,
					theDisplacement.y,
					theStartTime + theDuration,
					false,	//	aTwoFingerGestureFlag
					true);	//	aFlickGestureFlag

		MouseUp(	md,
					theStartTime + theDuration,
					false,	//	aTwoFingerGestureFlag
					true,	//	aFlickGestureFlag
					false);	//	aTouchSequenceWasCancelled

		[itsModel unlockModelData:&md];

#ifdef SCIENCE_CENTER_VERSION
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wundeclared-selector"
	//	The standard Geometry Games code would keep a weak reference
	//	to its view controller as itsDelegate, and declare any needed methods
	//	as part of a delegate protocol, but for this special-purpose version
	//	I think the following kludge is OK.
		[[[self window] rootViewController] performSelector:@selector(userIsStillPlaying) withObject:nil afterDelay:0.0];
#pragma clang diagnostic pop
#endif
	}
}


#pragma mark -
#pragma mark locations


- (CGPoint)touchLocation:(UITouch *)aTouch
{
	return [self pointLocation:[aTouch locationInView:self]];
}

- (CGPoint)gestureLocation:(UIGestureRecognizer *)aGestureRecognizer
{
	return [self pointLocation:[aGestureRecognizer locationInView:self]];
}

- (CGPoint)pointLocation:(CGPoint)aLocationInView	//	aLocationInView is in view coordinates
{
	CGRect	theViewBounds;
	CGPoint	theTouchPoint;
	
	//	Convert aLocationInView from view coordinates
	//	to intrinsic [-0.5, +0.5] coordinates.
	
	//	Both aLocationInView and theViewBounds should be in points (not pixels),
	//	but either would be fine just so everything is consistent.

	//	We expect the width and the height to be equal.
	theViewBounds = [self bounds];

	//	touch location in coordinates (0,0) to (width, height)
	theTouchPoint.x = aLocationInView.x - theViewBounds.origin.x;
	theTouchPoint.y = aLocationInView.y - theViewBounds.origin.y;

	//	touch location in coordinates (-width/2, -height/2) to (+width/2, +height/2)
	theTouchPoint.x -= 0.5 * theViewBounds.size.width;
	theTouchPoint.y -= 0.5 * theViewBounds.size.height;

	//	touch location in (-0.5, -0.5) to (+0.5, +0.5) coordinates
	theTouchPoint.x /= theViewBounds.size.width;
	theTouchPoint.y /= theViewBounds.size.height;
	
	//	Flip from iOS's top-down coordinates
	//	to the Torus Games' bottom-up coordinates.
	theTouchPoint.y = - theTouchPoint.y;
	
	return theTouchPoint;
}

- (CGPoint)touchMotion:(UITouch *)aTouch
{
	CGSize	theViewSize;
	CGPoint	theTouchPointA,
			theTouchPointB,
			theDifference;
	
	//	Compute how far the touch has moved since the previous touchesMoved event.

	//	We expect theViewSize.width and theViewSize.height to be equal.
	theViewSize = [self bounds].size;

	//	touch locations in view coordinates (0,0) to (theViewSize.width, theViewSize.height)
	theTouchPointA = [aTouch previousLocationInView:self];
	theTouchPointB = [aTouch locationInView:self];
	
	//	the relative motion, in the same coordinates
	theDifference.x = theTouchPointB.x - theTouchPointA.x;
	theDifference.y = theTouchPointB.y - theTouchPointA.y;

	//	the relative motion, in normalized coordinates
	theDifference.x /= theViewSize.width;
	theDifference.y /= theViewSize.height;

	//	Flip from iOS's top-down coordinates
	//	to the Torus Games' bottom-up coordinates.
	theDifference.y = - theDifference.y;
	
	return theDifference;
}


@end
